/* This file implements standard programs for depth shadow mapping. 
   These particular ones are suitable for additive lighting models, and
   include 3 techniques to reduce depth fighting on self-shadowed surfaces,
   constant bias, gradient (slope-scale) bias, and a fuzzy shadow map comparison*/

// Shadow caster vertex program.
void casterVP(
	float4 position			: POSITION,
	out float4 outPos		: POSITION,
	out float outDepth		: TEXCOORD0,

	uniform float4x4 worldViewProj,
	uniform float4 texelOffsets,
	uniform float4 depthRange
	)
{
	outPos = mul(worldViewProj, position);

	// fix pixel / texel alignment
	outPos.xy += texelOffsets.zw * outPos.w;
	// linear depth storage
	// offset / scale range output
	
	outDepth = (outPos.z - depthRange.x) * depthRange.w * 0.25;
}


// Shadow caster fragment program for high-precision single-channel textures	
void casterFP(
	float depth			: TEXCOORD0,
	out float4 result		: COLOR)
	
{
	// just smear across all components 
	// therefore this one needs high individual channel precision
	result = float4(depth, depth, depth, 1);
}


void receiverVP(
	float4 position		: POSITION,
	float4 normal		: NORMAL,

	out float4 outPos			: POSITION,
	out float4 outColour		: COLOR,
	out float4 outShadowUV		: TEXCOORD0,
	out float4 lightVec			: TEXCOORD1,

	uniform float4x4 world,
	uniform float4x4 worldIT,
	uniform float4x4 worldViewProj,
	uniform float4x4 texViewProj,
	uniform float4 lightPosition,
	uniform float4 lightColour,
	uniform float4 shadowDepthRange
	)
{
	float4 worldPos = mul(world, position);
	outPos = mul(worldViewProj, position);

	float3 worldNorm = mul(worldIT, normal).xyz;

	// calculate lighting (simple vertex lighting)
	float3 lightDir = normalize(
		lightPosition.xyz -  (worldPos.xyz * lightPosition.w));

	outColour = lightColour * max(dot(lightDir, worldNorm), 0.0);
	
	lightVec = lightPosition - worldPos;

	// calculate shadow map coords
	outShadowUV = mul(texViewProj, worldPos);
	outShadowUV.z = (outShadowUV.z - shadowDepthRange.x) * shadowDepthRange.w;
}


void receiverFP(
	float4 position			: POSITION,
	float4 shadowUV			: TEXCOORD0,
	float4 lightVec			: TEXCOORD1,
	float4 vertexColour		: COLOR,
	
	uniform float4 spotlightDir,
	uniform float4 spotlightParams,
	uniform float4 shadowColor,
	uniform float inverseShadowmapSize,
	uniform float fadeOutLength,
	uniform sampler2D shadowMap : register(s0),
	
	out float4 result		: COLOR
	)
{	
   float2 kernel[9];
   // Generate the tecture co-ordinates for the specified depth-map size
   kernel[0] = float2(0.0f, 0.0f);
   kernel[1] = float2( -inverseShadowmapSize, -inverseShadowmapSize );
   kernel[2] = float2(  inverseShadowmapSize, -inverseShadowmapSize );
   kernel[3] = float2( -inverseShadowmapSize,  inverseShadowmapSize );
   kernel[4] = float2(  inverseShadowmapSize,  inverseShadowmapSize );
   //kernel[5] = float2( -inverseShadowmapSize, 0.0f );
   //kernel[6] = float2(  inverseShadowmapSize, 0.0f);
   //kernel[7] = float2( 0.0f, -inverseShadowmapSize );
   //kernel[8] = float2( 0.0f,  inverseShadowmapSize );
   
   
   // point on shadowmap
   shadowUV.xy = shadowUV.xy / shadowUV.w;
   
   	//float attenuation = 1;
	float cosangle = dot(normalize(-spotlightDir), normalize(lightVec));
	float strength = (cosangle - spotlightParams.x) / (spotlightParams.y - spotlightParams.x);
	strength = clamp(strength,0,1);
	float final = 0;
	for( int i = 0; i < 5; i++ )
	{
		float casterdepth = tex2D(shadowMap, shadowUV.xy + kernel[i]) * 4;
		float dist =  min(1, max((shadowUV.z - casterdepth) / fadeOutLength, 0));
		final += (casterdepth > shadowUV.z) ? 1 : strength + (1-strength)*dist;
	}
   
   // Get the average
   final /= 5.0f;
   float fininv = 1 - final;
   result = float4(fininv*shadowColor.x + final, fininv*shadowColor.y + final, fininv*shadowColor.z + final, 1);
}




/*
  Basic ambient lighting vertex program
*/
void ambientOneTexture_vp(float4 position : POSITION,
						  float2 uv		  : TEXCOORD0,
						  
						  out float4 oPosition : POSITION,
						  out float2 oUv	   : TEXCOORD0,
						  out float4 colour    : COLOR,

						  uniform float4x4 worldViewProj,
						  uniform float4 ambient)
{
	oPosition = mul(worldViewProj, position);
	oUv = uv;
	colour = ambient;
}
/*
  Basic ambient lighting vertex program
*/
void ambient_vp(float4 position : POSITION,
						  
				out float4 oPosition : POSITION,
				out float4 colour    : COLOR,
				
				uniform float4x4 worldViewProj,
				uniform float4 ambient)
{
	oPosition = mul(worldViewProj, position);
	colour = ambient;
}