/* This file implements standard programs for depth shadow mapping. 
   These particular ones are suitable for additive lighting models, and
   include 3 techniques to reduce depth fighting on self-shadowed surfaces,
   constant bias, gradient (slope-scale) bias, and a fuzzy shadow map comparison*/

// Shadow caster vertex program.
void casterVP(
	float4 position			: POSITION,
	out float4 outPos		: POSITION,
	out float outDepth		: TEXCOORD0,

	uniform float4x4 worldViewProj,
	uniform float4 texelOffsets,
	uniform float4 depthRange
	)
{
	outPos = mul(worldViewProj, position);

	// fix pixel / texel alignment
	outPos.xy += texelOffsets.zw * outPos.w;
	// linear depth storage
	// offset / scale range output
	
	outDepth = (outPos.z - depthRange.x) * depthRange.w * 0.25;
}


// Shadow caster fragment program for high-precision single-channel textures	
void casterFP(
	float depth			: TEXCOORD0,
	out float4 result		: COLOR)
	
{
	// just smear across all components 
	// therefore this one needs high individual channel precision
	result = float4(depth, depth, depth, 1);
}


void receiverVP(
	float4 position		: POSITION,
	float4 normal		: NORMAL,
	float4 uv			: TEXCOORD0,

	out float4 outPos			: POSITION,
	out float4 outColour		: COLOR,
	out float4 outShadowUV		: TEXCOORD0,
	out float4 lightVec			: TEXCOORD1,
	out float4 lightUV			: TEXCOORD2,

	uniform float4x4 world,
	uniform float4x4 worldViewProj,
	uniform float4x4 texViewProj,
	uniform float4 lightPosition,
	uniform float4 shadowDepthRange
	)
{
	float4 worldPos = mul(world, position);
	outPos = mul(worldViewProj, position);
	
	// calculate lighting (simple vertex lighting)
	float3 lightDir = normalize(lightPosition.xyz -  (worldPos.xyz * lightPosition.w));
	
	lightVec = lightPosition - worldPos;

	// calculate shadow map coords
	outShadowUV = mul(texViewProj, worldPos);
	outShadowUV.z = (outShadowUV.z - shadowDepthRange.x) * shadowDepthRange.w;
	//depth = outShadowUV.z * shadowDepthRange.z + shadowDepthRange.x;
	
	lightUV = uv;
}


void receiverFP(
	float4 position			: POSITION,
	float4 shadowUV			: TEXCOORD0,
	float4 lightVec			: TEXCOORD1,
	float4 lightUV			: TEXCOORD2,
	float4 vertexColour		: COLOR,
	
	uniform float4 spotlightDir,
	uniform float4 spotlightParams,
	uniform float4 shadowColor,
	uniform float inverseShadowmapSize,
	uniform float4 shadowDepthRange,
	uniform float4 attenuation,

	uniform sampler2D shadowMap : register(s0),
	uniform sampler2D lightMap : register(s1),
	
	out float4 result		: COLOR
	)
{	
   float2 kernel[4];
   // Generate the tecture co-ordinates for the specified depth-map size
   kernel[0] = float2( -inverseShadowmapSize, -inverseShadowmapSize );
   kernel[1] = float2(  inverseShadowmapSize, -inverseShadowmapSize );
   kernel[2] = float2( -inverseShadowmapSize,  inverseShadowmapSize );
   kernel[3] = float2(  inverseShadowmapSize,  inverseShadowmapSize );
     
   // point on shadowmap
   shadowUV.xy = shadowUV.xy / shadowUV.w;
   
   	//float attenuation = 1;
	float cosangle = dot(normalize(-spotlightDir), normalize(lightVec));
	float strength = clamp((cosangle - spotlightParams.x) / (spotlightParams.y - spotlightParams.x),0,1);
	float strengthRev = 1 - strength;
	float final = 0;
	for( int i = 0; i < 4; i++ )
	{
		float casterdepth = tex2D(shadowMap, shadowUV.xy + kernel[i]) * 4;
		float dist = (shadowUV.z - casterdepth) * shadowDepthRange.z; 
		final += (dist < 1) ? 1 : (strength + strengthRev * clamp(dist / attenuation.x, 0, 1)); //attenuation constant = fade-out-strength
	}
   
   float reduce = 1 - final * 0.25f;
   float4 lightColor = tex2D(lightMap, lightUV.xy);
   float4 reduction = shadowColor * lightColor * float4(reduce,reduce,reduce,0);
   result = float4(1,1,1,1) - reduction;
}




/*
  Basic ambient lighting vertex program
*/
void ambientOneTexture_vp(float4 position : POSITION,
						  float2 uv		  : TEXCOORD0,
						  
						  out float4 oPosition : POSITION,
						  out float2 oUv	   : TEXCOORD0,
						  out float4 colour    : COLOR,

						  uniform float4x4 worldViewProj,
						  uniform float4 ambient)
{
	oPosition = mul(worldViewProj, position);
	oUv = uv;
	colour = ambient;
}
/*
  Basic ambient lighting vertex program
*/
void ambient_vp(float4 position : POSITION,
						  
				out float4 oPosition : POSITION,
				out float4 colour    : COLOR,
				
				uniform float4x4 worldViewProj,
				uniform float4 ambient)
{
	oPosition = mul(worldViewProj, position);
	colour = ambient;
}