[[FX]]

// Samplers
sampler2D albedoMap;
sampler2D specularMap = sampler_state
{
	Texture = "textures/common/global_specular.tga";
};

samplerCube ambientMap = sampler_state
{
	Address = Clamp;
	Filter = Bilinear;
	MaxAnisotropy = 1;
};

// Uniforms
float4 specParams <
	string desc_a = "a: specular mask";
	string desc_b = "b: specular exponent";
> = {1.0, 16.0, 0, 0};

// Contexts
context SHADOWMAP
{
	VertexShader = compile GLSL VS_SHADOWMAP;
	PixelShader = compile GLSL FS_SHADOWMAP;
}

context LIGHTING
{
	VertexShader = compile GLSL VS_GENERAL;
	PixelShader = compile GLSL FS_LIGHTING;
	
	ZWriteEnable = true;
	BlendMode = Add;
}

context TRANSLUCENT
{
	VertexShader = compile GLSL VS_GENERAL;
	PixelShader = compile GLSL FS_AMBIENT;

	ZWriteEnable = true;
	BlendMode = Blend;
}

[[VS_GENERAL]]
// =================================================================================================

#include "shaders/utilityLib/vertCommon.glsl"

uniform mat4 viewProjMat;
uniform vec3 viewerPos;
attribute vec3 vertPos;
attribute vec2 texCoords0;
attribute vec3 normal;

varying vec4 pos, vsPos;
varying vec2 texCoords;

varying vec3 tsbNormal;
varying float fogFactor;

void main( void )
{

	// Calculate normal
	vec3 _normal = normalize( calcWorldVec( normal ) );

	// Calculate tangent and bitangent
	tsbNormal = _normal;

	// Calculate world space position
	pos = calcWorldPos( vec4( vertPos, 1.0 ) );

	vsPos = calcViewPos( pos );

	// Calculate texture coordinates and clip space position
	texCoords = texCoords0;
	gl_Position = viewProjMat * pos;

	// FOG
	float _dist = length(vsPos);
	if (  _dist >= gl_Fog.start )
	{
		fogFactor = clamp( ((_dist-gl_Fog.start)/gl_Fog.end) , 0.0 , 1.0);
	}
	else
	{
		fogFactor = 0.0;
	}
}

[[VS_SHADOWMAP]]
// =================================================================================================
	
#include "shaders/utilityLib/vertCommon.glsl"

uniform mat4 viewProjMat;
uniform vec4 lightPos;
attribute vec3 vertPos;
varying vec3 lightVec;
uniform vec4 customInstData[4];

void main( void )
{
	vec4 pos = calcWorldPos( vec4( vertPos, 1.0 ) );
	pos.y -= (1.0-customInstData[0].a) * 20.0;
	lightVec = lightPos.xyz - pos.xyz;
	gl_Position = viewProjMat * pos;
}
	
	
[[FS_SHADOWMAP]]
// =================================================================================================

uniform vec4 lightPos;
uniform float shadowBias;
varying vec3 lightVec;

void main( void )
{
	float dist = length( lightVec ) / lightPos.w;
	gl_FragDepth = dist + shadowBias;
	
	// Clearly better bias but requires SM 3.0
	//gl_FragDepth = dist + abs( dFdx( dist ) ) + abs( dFdy( dist ) ) + shadowBias;
}


[[FS_LIGHTING]]
// =================================================================================================

#include "shaders/utilityLib/fragLighting.glsl" 

uniform vec4 specParams;
uniform sampler2D albedoMap;
uniform sampler2D specularMap;
uniform vec4 customInstData[4];

varying vec4 pos, vsPos;
varying vec2 texCoords;
varying vec3 tsbNormal;
varying float fogFactor;

void main( void )
{
	vec3 newCoords = vec3( texCoords, 0 );
	
	// Flip texture vertically to match the GL coordinate system
	newCoords.t *= -1.0;

	vec4 albedo = texture2D( albedoMap, newCoords.st );
	
	vec3 normal = tsbNormal;

	vec3 newPos = pos.xyz;

	//gl_FragColor.rgb = calcPhongSpotLight( newPos, normalize( normal ), albedo.rgb, specParams.x, specParams.y, -vsPos.z, 0.3 );
	//SpecularMapping
	float specValue = texture2D( specularMap, newCoords.st ).r * specParams.x;
	gl_FragColor.rgb = calcPhongSpotLight( newPos, normalize( normal ), albedo.rgb, specValue, specParams.y, -vsPos.z, 0.3 );

	// FOG
	gl_FragColor.rgb *= ( 1.0 - fogFactor );

	gl_FragColor.a = customInstData[0].a;
	gl_FragColor.r *= customInstData[0].a;
	gl_FragColor.g *= customInstData[0].a;
	gl_FragColor.b *= customInstData[0].a;

}


[[FS_AMBIENT]]	
// =================================================================================================

#include "shaders/utilityLib/fragLighting.glsl" 

uniform sampler2D albedoMap;
uniform samplerCube ambientMap;
uniform vec4 customInstData[4];

varying vec4 pos;
varying vec2 texCoords;
varying vec3 tsbNormal;
varying float fogFactor;

void main( void )
{
	/*
	vec3 newCoords = vec3( texCoords, 0 );
	// Flip texture vertically to match the GL coordinate system
	newCoords.t *= -1.0;
	vec4 albedo = texture2D( albedoMap, newCoords.st );
	vec3 normal = tsbNormal;
	gl_FragColor.rgb = albedo.rgb * textureCube( ambientMap, normal ).rgb;
	*/

	gl_FragColor = customInstData[0];

	// FOG
	gl_FragColor.rgb += ( gl_Fog.color.rgb - gl_FragColor.rgb ) * fogFactor;
}
