struct VS_OUTPUT
{
	float4	Position	: POSITION;		// Position in Screenspace
	float4	Diffuse		: COLOR0;		// Diffuse Light
	float3	ClampLight	: COLOR1;		// Color of Clamped Light
	float2	TexCoord	: TEXCOORD0;
	float3	WorldPos	: TEXCOORD1;
	float3	Normal		: TEXCOORD2;
	float3	ViewDir		: TEXCOORD3;
};



texture	ColorTexture;
sampler	ColorTextureSampler=sampler_state
{
	Texture		=<ColorTexture>;
	MinFilter	=Linear;
	MagFilter	=Linear;
	MipFilter	=Linear;
	MaxAnisotropy=8;
	AddressU	=WRAP;
	AddressV	=WRAP;
	AddressW	=WRAP;
};

texture	SpecularTexture;
sampler	SpecularTextureSampler=sampler_state
{
	Texture		=<SpecularTexture>;
	MinFilter	=Linear;
	MagFilter	=Linear;
	MipFilter	=Linear;
	MaxAnisotropy=8;
	AddressU	=WRAP;
	AddressV	=WRAP;
	AddressW	=WRAP;
};

texture	ReflectionTexture;
sampler	ReflectionTextureSampler=sampler_state
{
	Texture		=<ReflectionTexture>;
	MinFilter	=Linear;
	MagFilter	=Linear;
	MipFilter	=Linear;
	MaxAnisotropy=8;
	AddressU	=WRAP;
	AddressV	=WRAP;
	AddressW	=WRAP;
};

texture	EnvironmentTexture;
samplerCUBE	EnvironmentTextureSampler=sampler_state
{
	Texture		=<EnvironmentTexture>;
	MinFilter	=Linear;
	MagFilter	=Linear;
	MipFilter	=Linear;
	MaxAnisotropy=8;
};



// Bone world matrices
static const int MAX_MATRICES = 25;	
float4x3 BoneWorld[MAX_MATRICES];
int NumBones;

float4x4	WorldViewProjection;
float4x4	ViewProjection;
float4x4	World;
float3		CameraPosition; // Camera position in world space

// Lights
float3	AmbientLight;

static const int	MAX_LIGHTS=4;
int		NumLights;
float3	LightPosNrml[MAX_LIGHTS];	// Light's Position for Pointlights (LightRange>0.0) or Light's Normal for directional Lights (LightRange==0.0)
float3	LightColor[MAX_LIGHTS];		// Light's Color
float	LightRange[MAX_LIGHTS];		// Range of Light and Flag if it is a Pointlight (>0.0) or directional Light (==0.0)

float3	SpecPosNrml;				// For Specular Highlight: Light's Position for Pointlights (LightRange>0.0) or Light's Normal for directional Lights (LightRange==0.0)
float4	SpecColor;					// For Specular Highlight: Light's Color
float	SpecRange;					// For Specular Highlight: Range of Light and Flag if it is a Pointlight (>0.0) or directional Light (==0.0)

// Lightsource 0, clamped to a y-range
float3	ClampLightPosNrml;			// For clamped Light: Light's Position for Pointlights (LightRange>0.0) or Light's Normal for directional Lights (LightRange==0.0)
float4	ClampLightColor;			// For clamped Light: Light's Color
float	ClampLightRange;			// For clamped Light: Range of Light and Flag if it is a Pointlight (>0.0) or directional Light (==0.0)
float	ClampLightHeight;			// For clamped Light: Center of maximum Intensity
float	ClampLightWidth;			// For clamped Light: Width of Decay

// Fog
float4	FogColor;
float	FogBaseHeight;
float	FogHeight;
float	GroundFogWeight;

// Water
float	WaterClampHeight;
float	WaterClampInverse;

// Alphablending
int		AlphaPass;		// 0: No Alpha, 1: Alpha>0.5 (ZWrites enabled), 2: Alpha<0.5 (ZWrites disabled)
int		InvertNormal;	// Invert Normals (Used to paint transparent Backfaces)

// DebugDisplay
float4	DbgNormalFactor;













float3 TransformPosition(const float4 Position : POSITION0, const float3 BlendWeights : BLENDWEIGHT0, const float BlendDelta, const float4 BlendIndices : BLENDINDICES0)
{
	// calculate the position using the "normal" weights
	// and accumulate the weights to calculate the last weight
	float3	position =mul(Position, BoneWorld[BlendIndices.x])*BlendWeights[0];
	position	 	+=mul(Position, BoneWorld[BlendIndices.y])*BlendWeights[1];
	position		+=mul(Position, BoneWorld[BlendIndices.z])*BlendWeights[2];
	position		+=mul(Position, BoneWorld[BlendIndices.w])*BlendDelta;

	return position;
}



float3 TransformNormal(const float3 Normal : NORMAL0, const float3 BlendWeights : BLENDWEIGHT0, const float BlendDelta, const float4 BlendIndices : BLENDINDICES0)
{
	// calculate the normal using the "normal" weights
	// and accumulate the weights to calculate the last weight
	float3	normal	 =mul(Normal, BoneWorld[BlendIndices.x])*BlendWeights[0];
	normal			+=mul(Normal, BoneWorld[BlendIndices.y])*BlendWeights[1];
	normal			+=mul(Normal, BoneWorld[BlendIndices.z])*BlendWeights[2];
	normal			+=mul(Normal, BoneWorld[BlendIndices.w])*BlendDelta;

	if(InvertNormal)
		normal=-normal;

	return normalize(normal);
}



float3 CalcDiffuseLight(const float3 Position, const float3 Normal, uniform int NumLights, uniform int SpecLightMode)
{
	float3	diffuse=AmbientLight;
	float	intensity;

	for(int i=0; i<NumLights; i++)
	{
		if(LightRange[i]>0.0f)
		{
			float3	dir=Position-LightPosNrml[i];
			float	l=length(dir);
			intensity=dot(Normal, normalize(dir))*LightRange[i]/l;
		}
		else
			intensity=dot(Normal, LightPosNrml[i]);
		diffuse+=LightColor[i]*saturate(intensity);
	}

	if(SpecRange>0.0f)
	{
		float3	dir=Position-SpecPosNrml;
		float	l=length(dir);
		intensity=dot(Normal, normalize(dir))*SpecRange/l;
	}
	else
	{
		intensity=dot(Normal, SpecPosNrml);
	}
	diffuse+=SpecColor*saturate(intensity);

	return diffuse;
}






float3 CalcDirectionalSpecularLight(const float4 SpecMap, const float3 Normal, const float3 ViewDir)
{
	float3	dir=SpecPosNrml;										// Get Directionvector from Lightsource to Point
	float	intensity=SpecColor.a*2.0f;
	float3	halfvec=normalize(dir+ViewDir);							// Get Half Vector between Direction of Light and View
	float	specval=saturate(dot(Normal, halfvec));					// Calculate Angle of Half Vector and Surfacenormal
	specval=saturate(pow(specval, SpecMap.a*255.0f));				// Raise to some power, defined by SpecMap and clamp to Range [0, 1]
	return SpecColor.rgb*SpecMap.rgb*specval*intensity;				// Return Specular Highlight's Color
}



float3 CalcPointSpecularLight(const float4 SpecMap, const float3 Position, const float3 Normal, const float3 ViewDir)
{
	float3	dir=Position-SpecPosNrml;								// Get Directionvector from Lightsource to Point
	float	intensity=saturate(SpecRange/length(dir))*SpecColor.a*2.0f;
	float3	halfvec=normalize(normalize(dir)+ViewDir);				// Get Half Vector between Direction of Light and View
	float	specval=saturate(dot(Normal, halfvec));					// Calculate Angle of Half Vector and Surfacenormal
	specval=saturate(pow(specval, SpecMap.a*255.0f));				// Raise to some power, defined by SpecMap and clamp to Range [0, 1]
	return SpecColor.rgb*SpecMap.rgb*specval*intensity;				// Return Specular Highlight's Color
}



float3 CalcDirectionalClampLight(const float3 WorldPos, const float3 Normal)
{
	float	intensity=dot(Normal, ClampLightPosNrml);
	float	lightval=abs(WorldPos.y-ClampLightHeight)/ClampLightWidth;	// Calculate Height relative to clamped Lights's Center
	lightval=saturate(1.0f-lightval);									// Clamp to Range [0, 1]
	return ClampLightColor*intensity*lightval*lightval;					// Return clamped Light's Color
}



float3 CalcPointClampLight(const float3 WorldPos, const float3 Normal)
{
	float3	dir=WorldPos-ClampLightPosNrml;								// Get Directionvector from Lightsource to Point
	float	intensity=dot(Normal, normalize(dir))*ClampLightRange/length(dir);
	float	lightval=abs(WorldPos.y-ClampLightHeight)/ClampLightWidth;	// Calculate Height relative to clamped Lights's Center
	lightval=saturate(1.0f-lightval);									// Clamp to Range [0, 1]
	return ClampLightColor*intensity*lightval*lightval;					// Return clamped Light's Color
}



float3 CalcReflection(const float4 RefMap, const float3 Normal, const float3 ViewDir)
{
	float3	refdir=reflect(ViewDir, Normal);

	// NOTE: We need to invert y to compensate a Bug in the Engine!
	// Normals passed to DX are wrong but since Lights are wrong too, the Errors aren't visible.
	// ...unless we do Environmentmapping of cause.
	// Due to timeconstraints I'm not allowed to fix the original problem and introduce this Hack!
	float3	env=texCUBE(EnvironmentTextureSampler, float3(-refdir.x, refdir.y, -refdir.z));

	return RefMap*env*RefMap.a;
}



float3 CalcGroundFog(const float3 WorldPos, const float3 Color)
{
	float	fogy=saturate((WorldPos.y-FogBaseHeight)/(FogHeight-FogBaseHeight));	// Scale Worldspace-Y for groundfog from [FogHeight, 0] to [0, 1] and clamp to [0, 1] Range
	float	groundfog=FogColor.a*(1.0f-fogy);										// Multiply fogintensity from map with intensity derived from world-y
	float	fog=lerp(FogColor.a, groundfog, GroundFogWeight);						// Use the given weight to blend between groundfog and "normal" fog
	return lerp(Color, FogColor.rgb, fog);											// Blend object's color and fog's color
}



















VS_OUTPUT SkinVS(	float4 position 	: POSITION0,
					float3 blendWeights : BLENDWEIGHT0,
					float4 blendIndices : BLENDINDICES0,
					float3 normal 		: NORMAL0,
					float3 texCoord 	: TEXCOORD0,
					uniform int NumLights,
					uniform int SpecLightMode		// 1: Directional, 2:Point, anything else: none
)
{
	// Transform Position and Normal by Bonematrices
	float	blenddelta=(1.0f-blendWeights[0]-blendWeights[1]-blendWeights[2]);
	float3	skinposition=TransformPosition(position, blendWeights, blenddelta, blendIndices);
	float3	skinnormal=TransformNormal(normal, blendWeights, blenddelta, blendIndices);

	// Calculate diffuse Lighting
	float3	diffuse=CalcDiffuseLight(skinposition, skinnormal, NumLights, SpecLightMode);

	// Set Output
	VS_OUTPUT	pixel;
	pixel.Position=mul(float4(skinposition.xyz, 1.0f), ViewProjection);
	pixel.Diffuse=float4(diffuse, 1);
	pixel.ClampLight=LightColor[0]*max(0.0f, dot(skinnormal, LightPosNrml[0]));
	pixel.TexCoord.xy=texCoord.xy;
	pixel.WorldPos=skinposition;
	pixel.Normal=skinnormal;
	pixel.ViewDir=skinposition-CameraPosition;

	return pixel;
}



float4 PS_Textured(	VS_OUTPUT pixel,
					uniform int ClampLightMode,		// 1: Directional, 2:Point, anything else: none
					uniform int SpecLightMode,		// 1: Directional, 2:Point, anything else: none
					uniform bool Reflect,
					uniform bool Fog
					) : COLOR
{
	float3	dir;
	float	intensity;

	clip(pixel.WorldPos.y*WaterClampInverse);	// Don't draw if below Waterline


	float3	diffuse=pixel.Diffuse;
	float3	normal=normalize(pixel.Normal);		// Cache normalized Normal
	float3	viewdir=normalize(pixel.ViewDir);	// Cache normalized ViewDir


	// Calculate clamped Light
	if(ClampLightMode==1)
		diffuse+=CalcDirectionalClampLight(pixel.WorldPos, normal);
	if(ClampLightMode==2)
		diffuse+=CalcPointClampLight(pixel.WorldPos, normal);


	// Get diffuse Color and adjust for Alphablending
	float4	col=tex2D(ColorTextureSampler, pixel.TexCoord.xy);
	if(AlphaPass==0)			// No Alphablending
		col.a=1.0f;
	else if(AlphaPass==1)		// Pass 1: Paint quite opaque Areas (ZWrites enabled)
		clip(col.a-0.5f);
	else						// Pass 2: Paint less opaque Areas too (ZWrites disabled)
		clip(col.a-0.001f);		// Less than 1/255
	col.rgb*=diffuse;


	// Calculate Specular Highlight
	if(SpecLightMode==1)
	{
		float4	spec=tex2D(SpecularTextureSampler, pixel.TexCoord.xy);
		col.rgb+=CalcDirectionalSpecularLight(spec, normal, viewdir);
	}
	if(SpecLightMode==2)
	{
		float4	spec=tex2D(SpecularTextureSampler, pixel.TexCoord.xy);
		col.rgb+=CalcPointSpecularLight(spec, pixel.WorldPos, normal, viewdir);
	}


	// Calculate Reflections of Environment
	if(Reflect)
	{
		float4	ref=tex2D(ReflectionTextureSampler, pixel.TexCoord.xy);
		col.rgb+=CalcReflection(ref, normal, viewdir);
	}
/*	float4	ref=tex2D(ReflectionTextureSampler, pixel.TexCoord.xy);
	float3	refdir=reflect(viewdir, normal);
	float3	env=texCUBE(EnvironmentTextureSampler, float3(-refdir.x, refdir.y, -refdir.z));
	col.rgb+=ref*env*ref.a;*/


	// Calculate Groundfog
	if(Fog)
		col.rgb=CalcGroundFog(pixel.WorldPos, col.rgb);
/*	float	fogy=saturate((pixel.WorldPos.y-FogBaseHeight)/(FogHeight-FogBaseHeight));	// Scale Worldspace-Y for groundfog from [FogHeight, 0] to [0, 1] and clamp to [0, 1] Range
	float	groundfog=FogColor.a*(1.0f-fogy);											// Multiply fogintensity from map with intensity derived from world-y
	float	fog=lerp(FogColor.a, groundfog, GroundFogWeight);							// Use the given weight to blend between groundfog and "normal" fog
	col.rgb=lerp(col.rgb, FogColor.rgb, fog);											// Blend object's color and fog's color*/

	return col;
}



float4 CC_TexturedPS(VS_OUTPUT pixel) : COLOR
{
	return float4(AmbientLight, 1.0f);
}



float4 PS_DbgNormals(VS_OUTPUT pixel) : COLOR
{
	float3	normal=normalize(pixel.Normal);
	normal.y=-normal.y;
	return float4((0.5f+normal)*DbgNormalFactor.xyz*0.5f, 1.0f);
}

float4 PS_DbgRefNormals(VS_OUTPUT pixel) : COLOR
{
	float3	normal=normalize(float3(-pixel.Normal.x, pixel.Normal.y, -pixel.Normal.z));
//	return float4((0.5f+normal)*DbgNormalFactor.xyz*0.5f, 1.0f);

	float3	refdir=-reflect(normalize(pixel.ViewDir), normal);
	return float4((0.5f+refdir)*DbgNormalFactor.xyz*0.5f, 1.0f);
}



int CurNumLights=1;
VertexShader vsArraySkinned[4*3]=
{
	compile vs_1_1 SkinVS(0, 0),
	compile vs_1_1 SkinVS(1, 0),
	compile vs_1_1 SkinVS(2, 0),
	compile vs_1_1 SkinVS(3, 0),
	compile vs_1_1 SkinVS(0, 1),
	compile vs_1_1 SkinVS(1, 1),
	compile vs_1_1 SkinVS(2, 1),
	compile vs_1_1 SkinVS(3, 1),
	compile vs_1_1 SkinVS(0, 2),
	compile vs_1_1 SkinVS(1, 2),
	compile vs_1_1 SkinVS(2, 2),
	compile vs_1_1 SkinVS(3, 2)
};


#define SkinnedVSPS(__ClampLightMode, __SpecLightMode, __Reflect, __Fog)	\
pass Pass0																	\
{																			\
	VertexShader	=vsArraySkinned[CurNumLights+__SpecLightMode*4];		\
	PixelShader		=compile ps_2_0 PS_Textured(__ClampLightMode, __SpecLightMode, __Reflect, __Fog);	\
}


technique Skinned						{ SkinnedVSPS(0, 0, false, false) }
technique SkinnedDirClamp				{ SkinnedVSPS(1, 0, false, false) }
technique SkinnedPntClamp				{ SkinnedVSPS(2, 0, false, false) }
technique SkinnedDirSpec				{ SkinnedVSPS(0, 1, false, false) }
technique SkinnedPntSpec				{ SkinnedVSPS(0, 2, false, false) }
technique SkinnedDirClampDirSpec		{ SkinnedVSPS(1, 1, false, false) }
technique SkinnedDirClampPntSpec		{ SkinnedVSPS(1, 2, false, false) }
technique SkinnedPntClampDirSpec		{ SkinnedVSPS(2, 1, false, false) }
technique SkinnedPntClampPntSpec		{ SkinnedVSPS(2, 2, false, false) }

technique SkinnedRef					{ SkinnedVSPS(0, 0, true, false) }
technique SkinnedDirClampRef			{ SkinnedVSPS(1, 0, true, false) }
technique SkinnedPntClampRef			{ SkinnedVSPS(2, 0, true, false) }
technique SkinnedDirSpecRef				{ SkinnedVSPS(0, 1, true, false) }
technique SkinnedPntSpecRef				{ SkinnedVSPS(0, 2, true, false) }
technique SkinnedDirClampDirSpecRef		{ SkinnedVSPS(1, 1, true, false) }
technique SkinnedDirClampPntSpecRef		{ SkinnedVSPS(1, 2, true, false) }
technique SkinnedPntClampDirSpecRef		{ SkinnedVSPS(2, 1, true, false) }
technique SkinnedPntClampPntSpecRef		{ SkinnedVSPS(2, 2, true, false) }

technique SkinnedFog					{ SkinnedVSPS(0, 0, false, true) }
technique SkinnedDirClampFog			{ SkinnedVSPS(1, 0, false, true) }
technique SkinnedPntClampFog			{ SkinnedVSPS(2, 0, false, true) }
technique SkinnedDirSpecFog				{ SkinnedVSPS(0, 1, false, true) }
technique SkinnedPntSpecFog				{ SkinnedVSPS(0, 2, false, true) }
technique SkinnedDirClampDirSpecFog		{ SkinnedVSPS(1, 1, false, true) }
technique SkinnedDirClampPntSpecFog		{ SkinnedVSPS(1, 2, false, true) }
technique SkinnedPntClampDirSpecFog		{ SkinnedVSPS(2, 1, false, true) }
technique SkinnedPntClampPntSpecFog		{ SkinnedVSPS(2, 2, false, true) }

technique SkinnedRefFog					{ SkinnedVSPS(0, 0, true, true) }
technique SkinnedDirClampRefFog			{ SkinnedVSPS(1, 0, true, true) }
technique SkinnedPntClampRefFog			{ SkinnedVSPS(2, 0, true, true) }
technique SkinnedDirSpecRefFog			{ SkinnedVSPS(0, 1, true, true) }
technique SkinnedPntSpecRefFog			{ SkinnedVSPS(0, 2, true, true) }
technique SkinnedDirClampDirSpecRefFog	{ SkinnedVSPS(1, 1, true, true) }
technique SkinnedDirClampPntSpecRefFog	{ SkinnedVSPS(1, 2, true, true) }
technique SkinnedPntClampDirSpecRefFog	{ SkinnedVSPS(2, 1, true, true) }
technique SkinnedPntClampPntSpecRefFog	{ SkinnedVSPS(2, 2, true, true) }



technique SkinnedConstColor
{
	pass p0
	{
		VertexShader 	= compile vs_1_1 SkinVS(0, 0);
		PixelShader 	= compile ps_2_0 CC_TexturedPS();
	}
}



technique SkinnedDbgNormals
{
	pass p0
	{
		VertexShader 	= compile vs_1_1 SkinVS(0, 0);
		PixelShader 	= compile ps_2_0 PS_DbgNormals();
	}
}

technique SkinnedDbgRefNormals
{
	pass p0
	{
		VertexShader 	= compile vs_1_1 SkinVS(0, 0);
		PixelShader 	= compile ps_2_0 PS_DbgRefNormals();
	}
}
