struct ShadedPixel
{
	float4 Position : 		POSITION0;
	float2 TexCoord : 		TEXCOORD0;
	float3 WorldPosition : 	TEXCOORD3;
	float3 WorldNormal	 : 	NORMAL0;
	float3 WorldTangent	 : 	BINORMAL;
	
};

struct VS_OUTPUT
{
    float4  Position: POSITION;
    float4  Diffuse	: COLOR0;
	float3  Light1	: COLOR1;
	float3  TexCoord: TEXCOORD0;	// Note: Real texturecoordinates are in TexCoord.xy. Worldspace-Y encoded in TexCoord.z by VS for groundfog and clamped lighting
	float3	Normal	: TEXCOORD1;
	float3	RefDir	: TEXCOORD2;
	float3	ViewDir	: TEXCOORD3;
};


texture ColorTexture;
sampler ColorTextureSampler = sampler_state {
    Texture   = <ColorTexture>;
    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
    MaxAnisotropy = 8;
	AddressU  = WRAP;
	AddressV  = WRAP;
	AddressW  = WRAP;
};

texture SpecularTexture;
sampler SpecularTextureSampler = sampler_state {
    Texture   = <SpecularTexture>;
    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
    MaxAnisotropy = 8;
	AddressU  = WRAP;
	AddressV  = WRAP;
	AddressW  = WRAP;
};



// Bone world matrices
static const int MAX_MATRICES = 25;	
float4x3 BoneWorld[MAX_MATRICES];
int NumBones;

float4x4 WorldViewProjection;
float4x4 ViewProjection;
float4x4 World;
float3 CameraPosition; // Camera position in world space

// Light positions in world space
static const int MAX_LIGHTS = 4;
float3 LightPosNrml[MAX_LIGHTS];
float3 LightColor[MAX_LIGHTS];
float LightRange[MAX_LIGHTS];
int NumLights;

// Lightsource 0, clamped to a y-range
float	ClampLightHeight;
float	ClampLightWidth;

// Material properties
float3 Color;
float DiffuseCoeff;

// Fog
float4	FogColor;
float	FogBaseHeight;
float	FogHeight;
float	GroundFogWeight;

// Water
float	WaterClampHeight;
float	WaterClampInverse;

int AlphaMode;

float4 TexturedPS(VS_OUTPUT pixel) : COLOR
{
	clip(pixel.TexCoord.z*WaterClampInverse);

	float3	diffuse=pixel.Diffuse;

	float	lightval=abs(pixel.TexCoord.z-ClampLightHeight)/ClampLightWidth;	// Calculate Height relative to clamped Lights's Center
	lightval=saturate(1.0f-lightval);							// Clamp to Range [0, 1]
	diffuse+=pixel.Light1*lightval*lightval;					// Add to Light from other Sources

	float4 col = tex2D(ColorTextureSampler, pixel.TexCoord.xy);

	if(AlphaMode==0)			// No Alphablending
		col.a=1.0;
	else if(AlphaMode==1)		// Pass 1: paint quite opaque Areas (ZWrites enabled)
		clip(col.a-0.5);
	else						// Pass 2: paint less opaque Areas too (ZWrites disabled)
		clip(col.a-0.001);		// Less than 1/255

	col*=float4(diffuse, 1.0);
	col.rgb*=DiffuseCoeff;

// FixMe: Inefficient!!!
	float	fogy=saturate((pixel.TexCoord.z-FogBaseHeight)/(FogHeight-FogBaseHeight));	// Scale Worldspace-Y for groundfog from [FogHeight, 0] to [0, 1] and clamp to [0, 1] Range
	float	groundfog=FogColor.a*(1.0f-fogy);					// Multiply fogintensity from map with intensity derived from world-y
	float	fog=lerp(FogColor.a, groundfog, GroundFogWeight);	// Use the given weight to blend between groundfog and "normal" fog
	col.rgb=lerp(col.rgb, FogColor.rgb, fog);					// Blend object's color and fog's color
//	col.rgb=lerp(col.rgb, FogColor.rgb, fog*fog);				// Blend object's color and fog's color

	float3	halfvec=normalize(pixel.RefDir+pixel.ViewDir);				// Get Half Vector between Directions of Light and View
	float4	spec=tex2D(SpecularTextureSampler, pixel.TexCoord.xy);
	float	specval=saturate(dot(normalize(pixel.Normal), halfvec));	// Calculate Angle of Half Vector and Surfacenormal
	specval=pow(specval, spec.a*255.0);									// Raise to some power, defined by SpecMap

	col.rgb+=saturate(pixel.Light1*spec.rgb*specval);					// Add Specular Highlight to Object's Color

	return col;
}


float4 CC_TexturedPS(VS_OUTPUT pixel) : COLOR
{
	float4 col;
	col.xyz = Color;
	col.w = 1.0f;
	return float4(col.xyz, col.w);
}


VS_OUTPUT SkinVS(
									float4 position 	: POSITION0,
									float3 blendWeights : BLENDWEIGHT0,
									float4 blendIndices : BLENDINDICES0,
									float3 normal 		: NORMAL0,
									float3 texCoord 	: TEXCOORD0,
									uniform int NumLights
)
{
	// calculate the pos/normal using the "normal" weights
	// and accumulate the weights to calculate the last weight
	float3 skinPosition = float3(0.0f, 0.0f, 0.0f);
	float3 skinNormal = float3(0.0f, 0.0f, 0.0f);
	float fW = (1.0f-blendWeights[0]-blendWeights[1]-blendWeights[2]);
	
	skinPosition 	+= mul(position, BoneWorld[blendIndices.x]) * blendWeights[0];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.x]) * blendWeights[0];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.y]) * blendWeights[1];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.y]) * blendWeights[1];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.z]) * blendWeights[2];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.z]) * blendWeights[2];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.w]) * fW;
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.w]) * fW;

	// normalize normal
	skinNormal = normalize(skinNormal);

	if(AlphaMode)
	{
		float3	viewDir=(skinPosition-CameraPosition);
		float	d=dot(viewDir, skinNormal);
		if(d<-0.1)
			skinNormal=-skinNormal;
		else if(d<0.1)
			skinNormal*=d*10;
	}

	float3 diffuse = Color;
	for (int i = 1; i < NumLights; ++i)
	{
		float	intensity;
		if(LightRange[i]>0.0)
		{
			float3	dir=skinPosition-LightPosNrml[i];
			float	l=length(dir);
			intensity=dot(skinNormal, normalize(dir))*LightRange[i]/l;
		}
		else
			intensity=dot(skinNormal, LightPosNrml[i]);
		diffuse+=LightColor[i]*clamp(intensity, 0.0, 1.0);
	}

	skinNormal=normalize(skinNormal);	// Required for specular highlights

	// transform position from world space into view and then projection space
	VS_OUTPUT pixel;
	pixel.Position=mul(float4(skinPosition.xyz, 1.0f), ViewProjection);
	pixel.Diffuse=float4(diffuse,1);
	pixel.Light1=LightColor[0]*max(0.0, dot(skinNormal, LightPosNrml[0]));
	pixel.TexCoord.xy=texCoord.xy;							// UV-Coordinates
	pixel.TexCoord.z=skinPosition.y;						// Worldspace-Y for groundfog/light
	pixel.Normal=skinNormal;
	pixel.RefDir=LightPosNrml[0];
	pixel.ViewDir=normalize(skinPosition-CameraPosition);

	return pixel;
}




float4 GlowTexturedPS(VS_OUTPUT pixel) : COLOR
{
	clip(pixel.TexCoord.z*WaterClampInverse);

	float3	diffuse=pixel.Diffuse;

	float4 col = tex2D(ColorTextureSampler, pixel.TexCoord.xy);

	if(AlphaMode==0)			// No Alphablending
		col.a=1.0;
	else if(AlphaMode==1)		// Pass 1: paint quite opaque Areas (ZWrites enabled)
		clip(col.a-0.5);
	else						// Pass 2: paint less opaque Areas too (ZWrites disabled)
		clip(col.a-0.001);		// Less than 1/255

	col*=float4(diffuse, 1.0);
	col.rgb*=DiffuseCoeff;

// FixMe: Inefficient!!!
	float	fogy=saturate((pixel.TexCoord.z-FogBaseHeight)/(FogHeight-FogBaseHeight));	// Scale Worldspace-Y for groundfog from [FogHeight, 0] to [0, 1] and clamp to [0, 1] Range
	float	groundfog=FogColor.a*(1.0f-fogy);					// Multiply fogintensity from map with intensity derived from world-y
	float	fog=lerp(FogColor.a, groundfog, GroundFogWeight);	// Use the given weight to blend between groundfog and "normal" fog
	col.rgb=lerp(col.rgb, FogColor.rgb, fog);					// Blend object's color and fog's color
//	col.rgb=lerp(col.rgb, FogColor.rgb, fog*fog);				// Blend object's color and fog's color

	float3	halfvec=normalize(pixel.RefDir+pixel.ViewDir);				// Get Half Vector between Directions of Light and View
	float4	spec=tex2D(SpecularTextureSampler, pixel.TexCoord.xy);
	float	specval=saturate(dot(normalize(pixel.Normal), halfvec));	// Calculate Angle of Half Vector and Surfacenormal
	specval=pow(specval, spec.a*255.0);									// Raise to some power, defined by SpecMap

	col.rgb+=saturate(pixel.Light1*spec.rgb*specval);					// Add Specular Highlight to Object's Color

	return col;
}

VS_OUTPUT GlowSkinVS(
									float4 position 	: POSITION0,
									float3 blendWeights : BLENDWEIGHT0,
									float4 blendIndices : BLENDINDICES0,
									float3 normal 		: NORMAL0,
									float3 texCoord 	: TEXCOORD0,
									uniform int NumLights
)
{
	// calculate the pos/normal using the "normal" weights
	// and accumulate the weights to calculate the last weight
	float3 skinPosition = float3(0.0f, 0.0f, 0.0f);
	float3 skinNormal = float3(0.0f, 0.0f, 0.0f);
	float fW = (1.0f-blendWeights[0]-blendWeights[1]-blendWeights[2]);
	
	skinPosition 	+= mul(position, BoneWorld[blendIndices.x]) * blendWeights[0];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.x]) * blendWeights[0];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.y]) * blendWeights[1];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.y]) * blendWeights[1];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.z]) * blendWeights[2];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.z]) * blendWeights[2];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.w]) * fW;
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.w]) * fW;

	// normalize normal
	skinNormal = normalize(skinNormal);

	if(AlphaMode)
	{
		float3	viewDir=(skinPosition-CameraPosition);
		float	d=dot(viewDir, skinNormal);
		if(d<-0.1)
			skinNormal=-skinNormal;
		else if(d<0.1)
			skinNormal*=d*10;
	}

	skinNormal=normalize(skinNormal);	// Required for specular highlights

	// transform position from world space into view and then projection space
	VS_OUTPUT pixel;
	pixel.Position=mul(float4(skinPosition.xyz, 1.0f), ViewProjection);
	pixel.Diffuse=float4(0.882, 0.858, 0.705, 1.0);
	pixel.Light1=0.0;
	pixel.TexCoord.xy=texCoord.xy;							// UV-Coordinates
	pixel.TexCoord.z=skinPosition.y;						// Worldspace-Y for groundfog/light
	pixel.Normal=skinNormal;
	pixel.RefDir=LightPosNrml[0];
	pixel.ViewDir=normalize(skinPosition-CameraPosition);

	return pixel;
}




int CurNumLights= 1;
VertexShader vsArray[4] = { compile vs_1_1 SkinVS(0),
                            compile vs_1_1 SkinVS(1),
                            compile vs_1_1 SkinVS(2),
                            compile vs_1_1 SkinVS(3)
                          };

technique Skinned
{
	pass p0
	{
		VertexShader 	= (vsArray[CurNumLights]);
		PixelShader 	= compile ps_2_0 TexturedPS();
		
		//FillMode = WireFrame;
		
	}
}


technique CC_Skinned
{
	pass p0
	{
		VertexShader 	= compile vs_1_1 SkinVS(0);
		PixelShader 	= compile ps_2_0 CC_TexturedPS();
		
		//FillMode = WireFrame;
		
	}
}


technique Glow_Skinned
{
	pass p0
	{
		VertexShader 	= compile vs_1_1 GlowSkinVS(0);
		PixelShader 	= compile ps_2_0 GlowTexturedPS();
		
		//FillMode = WireFrame;
		
	}
}
