struct ShadedPixel
{
	float4 Position : 		POSITION0;
	float2 TexCoord : 		TEXCOORD0;
	float3 WorldPosition : 	TEXCOORD3;
	float3 WorldNormal	 : 	NORMAL0;
	float3 WorldTangent	 : 	BINORMAL;
	
};

struct VS_OUTPUT
{
    float4  Position: POSITION;
    float4  Diffuse	: COLOR0;
	float3  Light1	: COLOR1;
	float3  TexCoord: TEXCOORD0;	// Note: Real texturecoordinates are in TexCoord.xy. Worldspace-Y encoded in TexCoord.z by VS for groundfog and clamped lighting
};


texture ColorTexture;
sampler ColorTextureSampler = sampler_state {
    Texture   = <ColorTexture>;
    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
    MaxAnisotropy = 8;
	AddressU  = WRAP;
	AddressV  = WRAP;
	AddressW  = WRAP;
};



// Bone world matrices
static const int MAX_MATRICES = 25;	
float4x3 BoneWorld[MAX_MATRICES];
int NumBones;

float4x4 WorldViewProjection;
float4x4 ViewProjection;
float4x4 World;
float3 CameraPosition; // Camera position in world space

// Light positions in world space
static const int MAX_LIGHTS = 4;
float3 LightNormal[MAX_LIGHTS];
float3 LightColor[MAX_LIGHTS];
int NumLights;

// Lightsource 0, clamped to a y-range
float	ClampLightHeight;
float	ClampLightWidth;

// Material properties
float3 Color;
float DiffuseCoeff;
float SpecularCoeff;
float SpecularPower;

// Fog
float4	FogColor;
float	FogBaseHeight;
float	FogHeight;
float	GroundFogWeight;

int AlphaMode;

float4 TexturedPS(VS_OUTPUT pixel) : COLOR
{
	float3	diffuse=pixel.Diffuse;

// Replace clamp(x, 0, 1) by saturate(x)

	float	lightval=abs(pixel.TexCoord.z-ClampLightHeight)/ClampLightWidth;
//	diffuse+=pixel.Light1*clamp(1.0f-lightval, 0, 1);
	lightval=clamp(1.0f-lightval, 0, 1);
	diffuse+=pixel.Light1*lightval*lightval;

	float4 col = tex2D(ColorTextureSampler, pixel.TexCoord.xy);

	if(AlphaMode==0)
		col.a=1.0;
	else if(AlphaMode==1)
		clip(col.a-0.5);
	clip(col.a-0.001);	// Less than 1/255

	col *= float4(diffuse, 1.0);
	col.xyz *= DiffuseCoeff;

// FixMe: Inefficient!!!
	float	fogy=(pixel.TexCoord.z-FogBaseHeight)/(FogHeight-FogBaseHeight);	// Scale Worldspace-Y for groundfog from [FogHeight, 0] to [0, 1]
	fogy=clamp(fogy, 0.0f, 1.0f);								// Clamp to [0, 1] Range
	float	groundfog=FogColor.a*(1.0f-fogy);					// Multiply fogintensity from map with intensity derived from world-y
	float	fog=lerp(FogColor.a, groundfog, GroundFogWeight);	// Use the given weight to blend between groundfog and "normal" fog
	col.rgb=lerp(col.rgb, FogColor.rgb, fog);					// Blend object's color and fog's color
//	col.rgb=lerp(col.rgb, FogColor.rgb, fog*fog);				// Blend object's color and fog's color
//	col.rgb=FogColor.rgb*FogColor.a;

	return col;
}


float4 CC_TexturedPS(VS_OUTPUT pixel) : COLOR
{
	float4 col;
	col.xyz = Color;
	col.w = 1.0f;
	return float4(col.xyz, col.w);
}


VS_OUTPUT SkinVS(
									float4 position 	: POSITION0,
									float3 blendWeights : BLENDWEIGHT0,
									float4 blendIndices : BLENDINDICES0,
									float3 normal 		: NORMAL0,
									float3 texCoord 	: TEXCOORD0,
									uniform int NumLights
)
{
	// calculate the pos/normal using the "normal" weights
	// and accumulate the weights to calculate the last weight
	float3 skinPosition = float3(0.0f, 0.0f, 0.0f);
	float3 skinNormal = float3(0.0f, 0.0f, 0.0f);
	float fW = (1.0f-blendWeights[0]-blendWeights[1]-blendWeights[2]);
	
	skinPosition 	+= mul(position, BoneWorld[blendIndices.x]) * blendWeights[0];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.x]) * blendWeights[0];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.y]) * blendWeights[1];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.y]) * blendWeights[1];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.z]) * blendWeights[2];
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.z]) * blendWeights[2];
	skinPosition 	+= mul(position, BoneWorld[blendIndices.w]) * fW;
	skinNormal 		+= mul(normal,   BoneWorld[blendIndices.w]) * fW;

	// normalize normal
	skinNormal = normalize(skinNormal);

	if(AlphaMode)
	{
		float3	viewDir=(skinPosition-CameraPosition);
		if(dot(viewDir, skinNormal)<0)
			skinNormal=-skinNormal;
	}

	float3 diffuse = Color;
	for (int i = 1; i < NumLights; ++i)
	{
		float intensity = max(0.0, dot(skinNormal, LightNormal[i]));
		diffuse += LightColor[i] * intensity;
	}

	// transform position from world space into view and then projection space
	VS_OUTPUT pixel;
	pixel.Position=mul(float4(skinPosition.xyz, 1.0f), ViewProjection);
	pixel.Diffuse=float4(diffuse,1);
	pixel.Light1=LightColor[0]*max(0.0, dot(skinNormal, LightNormal[0]));
	pixel.TexCoord.xy=texCoord.xy;							// UV-Coordinates
//	pixel.TexCoord.z=1-skinPosition.y/FogHeight;			// Worldspace-Y for groundfog, scale to [0, 1] from [FogHeight, 0]
//	pixel.TexCoord.z=clamp(pixel.TexCoord.z, 0.0f, 1.0f);	// Clamp to [0, 1] Range
	pixel.TexCoord.z=skinPosition.y;						// Worldspace-Y for groundfog/light

	return pixel;
}

int CurNumLights= 1;
VertexShader vsArray[4] = { compile vs_1_1 SkinVS(1),
                            compile vs_1_1 SkinVS(2),
                            compile vs_1_1 SkinVS(3),
                            compile vs_1_1 SkinVS(4)
                          };

technique Skinned
{
	pass p0
	{
		VertexShader 	= (vsArray[CurNumLights]);
		PixelShader 	= compile ps_2_0 TexturedPS();
		
		//FillMode = WireFrame;
		
	}
}


technique CC_Skinned
{
	pass p0
	{
		VertexShader 	= compile vs_1_1 SkinVS(0);
		PixelShader 	= compile ps_2_0 CC_TexturedPS();
		
		//FillMode = WireFrame;
		
	}
}
