#!/bin/bash
############################################################
# apkgsfx.sh	Self Extracting Installer
# 9/10/09	extractor piece, extract attached archive and call the
#		embedded installer that is specified in variables, values
#		will be replaced through token replacement from BDC.
# 10/19/09	allow non-archive attachment
# 12/14/09	when run without log file option (default) place log in /boot
#		and rename log to name of INSTALLER
############################################################

TRUE=1
FALSE=0
INSTALLER="$$Install$$"
ARGS="$$Args$$"
COMPRESSION="$$Ext$$"
QUIET="$$Quiet$$"
LOG="$$Log$$"
USER_ARG=""
TMPDIR=`mktemp -d /tmp/selfextract.XXXXXX`
DEFAULT_LOG="extractor.log"
LOG_DIR="/tmp"
WORK_LOG=""
VERBOSE=$TRUE
DEBUG=""

#---------------------------------------
## use for normal program output and recording in WORK_LOG
#  if debug out is enabled will display these messages also
function EchoLog()
{
  if [ "$VERBOSE" = "$TRUE" -o "$DEBUG" = "$TRUE" ]; then
    echo -e "$*"
  fi;
  echo "[extractor `date +\"%Y-%m-%d %H:%M:%S\"`] :$*" >> $WORK_LOG
}

#---------------------------------------
## use for debug program output and recording in WORK_LOG
function WriteLog()
{
  if [ "$DEBUG" = "$TRUE" ]; then
    echo -e "$*"
  fi;
  echo "[extractor `date +\"%Y-%m-%d %H:%M:%S\"`] :$*" >> $WORK_LOG
}

#---------------------------------------
function usage()
{
  echo -e "Self Extracting Installer\n"
  echo -e "  Usage: $0 [options] [args]\n"
  echo -e "         [options]"
  echo -e "         -d | --debug  display debug info"
  echo -e "         -q | --quiet  do not display status info"
  echo -e "         -h | --help   display this help\n"
  echo -e "         [args]"
  echo -e "         user supplied args to be passed to"
  echo -e "         the embedded installer\n"
}

#===========================================================
archive=`awk '/^__ARCHIVE_BELOW__/ {print NR + 1; exit 0; }' "$0"`

while [ "$1" != "" ]; do
  case $1 in
    -d | --debug )   DEBUG=$TRUE
                     USER_ARG="$USER_ARG --debug"
                     ;;
    -h | --help )    usage
                     exit
                     ;;
    -l | --log )     shift
                     LOG=$1
                     ;;
    -q | --quiet )   VERBOSE=$FALSE
                     USER_ARG="$USER_ARG --quiet"
                     ;;
    *)               USER_ARG="$USER_ARG $1"
  esac;
  shift
done;

## create default LOG variable to check if it has been replaced
#  if empty use installer name and leave in /boot
if [ -z "$LOG" ]; then
  if [ -d /boot ]; then
    T_LOG=`echo $INSTALLER | sed 's/\..*$//'`
    WORK_LOG="/boot/$T_LOG.log"
    if [ -e "$WORK_LOG" ]; then
      rm -f "$WORK_LOG"
    fi;
    USER_ARG="$USER_ARG --log $T_LOG.log"
  else
    WORK_LOG="$LOG_DIR/$DEFAULT_LOG"
    USER_ARG="$USER_ARG --log $DEFAULT_LOG"
  fi;
else
  WORK_LOG="$LOG_DIR/$LOG"
  USER_ARG="$USER_ARG --log $LOG"
fi;

EchoLog "Self Extracting Installer\n"

WriteLog " - using log file $WORK_LOG"

case $COMPRESSION in
  gz | GZ | tgz | TGZ)    WriteLog " - gzip archive handling enabled"
                          ;;
  bz2 | BZ2)  WriteLog " - bzip2 archive handling enabled"
              ;;
  tar | TAR)  WriteLog " - tar archive handling enabled"
              ;;
  *)          WriteLog " - archive handling disabled"
esac;

EchoLog " - extracting archive to $TMPDIR"

case $COMPRESSION in
  gz | GZ | tgz | TGZ)    tail -n+$archive "$0" | tar xvz -C "$TMPDIR"
                          ;;
  bz2 | BZ2)  tail -n+$archive "$0" | tar xvj -C "$TMPDIR"
              ;;
  tar | TAR)  tail -n+$archive "$0" | tar xv -C "$TMPDIR"
              ;;
  *)          INSTALLER=`basename $INSTALLER`
              tail -n+$archive "$0" > $TMPDIR/$INSTALLER
esac;

if [ -f "$TMPDIR/$INSTALLER" ]; then
  if [ ! -x "$TMPDIR/$INSTALLER" ]; then
    WriteLog " - making installer executable"
    chmod 700 "$TMPDIR/$INSTALLER"
  fi;
  EchoLog " - spawning embedded installer '$TMPDIR/$INSTALLER $ARGS $USER_ARG'"
  eval $TMPDIR/$INSTALLER $ARGS $USER_ARG
  rc=$?
else
  EchoLog "Installer $INSTALLER not found, unable to continue\n\a\n"
  EchoLog " extractor exit status *failure*"
  exit 6
fi;

rm -rf $TMPDIR

WriteLog " extractor exit status *successful*"
WriteLog " installer exit code $rc"

exit $rc

__ARCHIVE_BELOW__ 
