/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2016 Adobe Systems Incorporated                          */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE:  All information contained herein is, and remains           */
/* the property of Adobe Systems Incorporated and its suppliers,       */
/* if any.  The intellectual and technical concepts contained          */
/* herein are proprietary to Adobe Systems Incorporated and its        */
/* suppliers and are protected by all applicable intellectual property */
/* laws, including trade secret and copyright laws.                    */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe Systems Incorporated.                                    */
/*                                                                     */
/***********************************************************************/

function WorkflowDisplayAdapter(/*[PanelSurface]*/ inSurface, /*[Workflow]*/ inWorkflow)
{
	throwInvalid(inSurface);
	
	var workflow = inWorkflow;
	var surface = inSurface;
	var contentStorage = (isValidProperty(WorkflowDisplayAdapter.sContentStorage) ? WorkflowDisplayAdapter.sContentStorage : IContentStorage.create());
	WorkflowDisplayAdapter.sContentStorage = contentStorage;
	
	throwInvalid(contentStorage);
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show selected workflow
	//
	this.showInitial = function()
	{
		hideStepIndicator();
		displayWorkflowContent();
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show next step of Workflow
	//
	this.showStep = function(/*[Number]*/ inStepNumber)
	{
		hideStepIndicator();
		displayWorkflowStepContent(inStepNumber);
		
		try
		{
			showStepIndicator(inStepNumber);
		}
		catch(exc)
		{
			exclog(exc);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show workflow step success
	//
	this.showStepSuccess = function(/*[Number]*/ inCurrentStep)
	{
		hideStepIndicator();
		displayWorkflowStepContent(inCurrentStep);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show workflow step failure
	//
	this.showFailure = function(/*[Number]*/ inCurrentStep, /*[WorkflowContent]*/ inContentXtra)
	{
		hideStepIndicator();
		displayWorkflowStepContent(inCurrentStep, inContentXtra);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show workflow processing success
	//
	this.showSuccess = function()
	{
		hideStepIndicator();
		displayWorkflowContent();
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show workflow processing canceled
	//
	this.showCancel = function()
	{
		hideStepIndicator();
		displayWorkflowContent();
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Show all workflow titles
	//
	this.showTitles = function(/*[String]*/ inLocatorID, /*[Array]*/ inWorkflows, /*[Function]*/ inClickHandler)
	{
		throwInvalid(inLocatorID);
		throwInvalid(inWorkflows, 'Array');
		
		hideStepIndicator();

		var workflows = inWorkflows;
		var locatorID = inLocatorID;
		var clickHandler = inClickHandler;
		var list = '';
		var i = 0;

		function doDisplay()
		{
			list = '<ul style="list-style-type:none">' + list + '</ul>';
			display(inLocatorID, list);
		}

		function getWorkflowTitle()
		{
			if (i < workflows.length)
			{
				var content = null;
				var workflow = workflows[i];

				try
				{
					content = workflow.getContent(CONTENTLOCATOR_WORKFLOW_LIST_TITLE);
				}
				catch (exc)
				{
					exclog(exc);
					content = null;
				}

				i++;

				if (content)
				{
					generateWorkflowHtml(workflow, content, inClickHandler, function (contentHtml)
					{
						if (contentHtml)
						{
							contentHtml = '<li>' + contentHtml + '</li>';
							list += contentHtml;
						}

						getWorkflowTitle();
					});
				}
				else
				{
					getWorkflowTitle();
				}
			}
			else
			{
				doDisplay();
			}
		}

		getWorkflowTitle();
	}
	
	// private ///////////////////////////////////////////////////////////////////
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Take all Workflow content locator id's and try to display them on the 
	// current surface.
	//
	function displayWorkflowContent()
	{
		if (isValidProperty(workflow))
		{
			displayAllContent(workflow);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Take all Workflow step content locator id's and try to display them on the 
	// current surface.
	//
	function displayWorkflowStepContent(/*[Number]*/ inStepNumber, /*[WorkflowContent]*/ inContentXtra)
	{
		if (isValidProperty(workflow))
		{
			var step = null;
			
			try
			{
				step = workflow.getStep(inStepNumber);
			}
			catch(exc)
			{
				exclog(exc);
				step = null;
			}
		
			if (step)
			{
				displayAllContent(step);
			}
			
			if (isValidProperty(inContentXtra))
			{
				displayAllContent(inContentXtra);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Try to display all content of the content owner
	// inContentOwner could be of type Workflow, WorkflowStep, WorkflowContent
	//
	function displayAllContent(/*[Object]*/ inContentOwner)
	{
		throwInvalid(inContentOwner);
		throwInvalid(inContentOwner.getContentLocator);
		throwInvalid(inContentOwner.getContent);

		progress(true);

		var contentOwner = inContentOwner;
		var locators = null;
		var count = 0;
		
		try
		{
			locators = inContentOwner.getContentLocator();
		}
		catch(exc)
		{
			exclog(exc);
			locators = null;
		}
		
		throwInvalid(locators, 'Array');

		count = locators.length;

		function doDisplay(/*[String]*/ inLocator)
		{
			var locator = inLocator;
			var content = null;

			try
			{
				content = contentOwner.getContent(locator);
			}
			catch(exc)
			{
				exclog(exc);
				content = null;
			}

			count--;

			if (content)
			{
				generateWorkflowHtml(workflow, content, null, function(contentHtml)
				{
					display(locator, contentHtml);

					if (count <= 0)
					{
						progress(false);
					}
				});
			}
		}

		forEach(locators, function(locator)
		{
			doDisplay(locator);
		});
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Generate HTML for workflow content
	//
	function generateWorkflowHtml(/*[Workflow]*/ inWorkflow, /*[WorkflowContent]*/ inContent, /*[Function]*/ inClickHandler, /*[Function]*/ inCallback)
	{
		var ret = '';
		var workflow = inWorkflow;
		var mimeType = inContent.mimeType;
		var clickHandler = inClickHandler;
		var callback = inCallback;

		function addClickHdl(/*[String]*/ inHTML)
		{
			var html = inHTML;

			if (isValidProperty(html) && html.length && isValidProperty(clickHandler))
			{
				html = '<a id="' + workflow.id + '" href="#" onclick="' + clickHandler.name + '(this.id);">' + html + '</a>';
			}

			callback(html);
		}

		switch(mimeType)
		{
			case 'text/html':
			{
				contentStorage.getRaw(inWorkflow.id, inContent.id, function(html)
				{
					if (isValidProperty(html) && html.length)
					{
						replaceWorkflowStringSymbols(html, contentStorage, workflow.id, function(html)
						{
							replaceWorkflowURLSymbols(html, contentStorage, workflow.id, function(html)
							{
								addClickHdl(html);
							});
						});
					}
					else
					{
						addClickHdl(html);
					}
				});
			}
			break;
			
			case 'text/plain':
			{
				contentStorage.getString(inWorkflow.id, inContent.id, function(content)
				{
					if (isValidProperty(content) && content.length && !isValidProperty(clickHandler))
					{
						content = generateHtml(content, mimeType);
					}

					addClickHdl(content);
				});
			}
			break;
			
			case 'audio/x-mpeg':
			case 'audio/x-wav':
			case 'image/gif':
			case 'image/jpeg':
			case 'image/png':
			case 'video/mp4':
			{
				contentStorage.getURL(inWorkflow.id, inContent.id, function(url)
				{
					var html = null;
					if (isValidProperty(url))
					{
						html = generateHtml(url, mimeType);
					}

					addClickHdl(html);
				});
			}
			break;

			default:
				callback(null);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Generate HTML for content
	//
	function generateHtml(/*[String]*/ inContent, /*[String]*/ inMimeType)
	{
		var ret = inContent;
		
		if (isValidProperty(ret))
		{
			switch(inMimeType)
			{
				case 'text/plain':
				{
					if (ret.length)
					{
						ret = '<p>' + ret + '</p>';
					}
				}
				break;
				case 'audio/x-mpeg':
				case 'audio/x-wav':
				{
					if (ret.length)
					{
						ret  = '<audio autoplay><source src="' + ret;
						ret += '" type="';
						ret += inMimeType;
						ret += '"></audio>';
					}
				}
				break;
				case 'image/gif':
				case 'image/jpeg':
				case 'image/png':
				{
					if (ret.length)
					{
						ret  = '<img src="' + ret;
						ret += '"/>';
					}
				}
				break;
				case 'video/mp4':
				{
					if (ret.length)
					{
						ret  = '<video autoplay loop><source src="' + ret;
						ret += '" type="';
						ret += inMimeType;
						ret += '"></video>';
					}
				}
				break;
			}
		}
				
		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Show/Hide progress
	//
	function progress(/*[Boolean]*/ inVisible)
	{
		if (inVisible)
		{
			var snippet = '<img src="./surfaces/progress.gif">';
			surface.displayByID(CONTENTLOCATOR_PROGRESS, snippet);

		}
		else
		{
			surface.displayByID(CONTENTLOCATOR_PROGRESS, '');

		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Display content at locator ID
	//
	function display(/*[String]*/ inLocator, /*[String]*/ inContentHtml)
	{
		surface.displayByID(inLocator, inContentHtml);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Show indicator overlay
	//
	function showStepIndicator(/*[Number]*/ inStepNumber)
	{
		if (isValidProperty(workflow))
		{
			var step = workflow.getStep(inStepNumber)
			throwInvalid(step, 'WorkflowStep');
			
			var indicator = step.getAllIndicator();
			throwInvalid(indicator, 'Array');
			
			for (var i=0; i<indicator.length; i++)
			{
				indicator[i].display(contentStorage);
			}
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Hide indicator overlay
	//
	function hideStepIndicator()
	{
		if (isValidProperty(workflow))
		{
			for (var s=0; s<workflow.getStepLength(); s++)
			{
				var step = workflow.getStep(s)
				if (isValidProperty(step))
				{
					var indicator = step.getAllIndicator();
					if (isValidProperty(indicator, 'Array'))
					{
						for (var i = 0; i < indicator.length; i++)
						{
							indicator[i].cancelDisplay();
						}
					}
				}
			}
		}

		var cs = new CSInterface();
		cs.evalScript('hideAllIndicator();');
	}
	
}

// global content storage
//
WorkflowDisplayAdapter.sContentStorage = null;